/*
 * Toolkit GUI, an application built for operating pinkRF's signal generators.
 *
 * Contact: https://www.pinkrf.com/contact/
 * Copyright © 2018-2024 pinkRF B.V
 * GNU General Public License version 3.
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License along with this program. If not, see https://www.gnu.org/licenses/
 *
 * Author: Iordan Svechtarov
 */

#include "rf_system_minimal.h"
#include <QCoreApplication>
#include <QDebug>
#include <QMessageBox>


//
//TODO:
//it would probably be best if the constructor class were static as well.
//


RF_System_Minimal::RF_System_Minimal()
{
	qDebug() << "RF_System_minimal: [Reading config file]";
	config = new ConfigHandler(QCoreApplication::applicationDirPath() + "/config.txt");

	//Set up a single port for RCM;
	RCM_USB_port = new RCM_Class();
	RCM_USB_port->Setup(config->get_RCM_USB_port());
	connect(RCM_USB_port->rcm_port, &QSerialPort::readyRead, this, &RF_System_Minimal::RCM_USB_message_handler);		//Live handler
//	connect(RCM_USB_port->rcm_port, &QSerialPort::readyRead, this, &RF_System_Minimal::RCM_USB_message_handler_blind);
//	connect(RCM_USB_port->serial, &QSerialPort::errorOccurred, this, &RF_System_Minimal::RCM_USB_error_handler);

	threadList = new QList<QThread*>();

	for (int i = 0; i < channel_count; i++)
	{
		channel = new RF_Channel(i+1, i+1, config->get_SGX_port(i+1));
		RF_System::Channels->append(channel);

		connect(this, &RF_System_Minimal::signal_RCM_USB_message, RF_System::Channels->at(i), &RF_Channel::RCM_Live_serial_writeRead_to_SGx);
		connect(this, &RF_System_Minimal::signal_RCM_sweep_message, RF_System::Channels->at(i), &RF_Channel::RCM_Live_serial_Sweep_handler);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_RCM_serial_response, this, &RF_System_Minimal::RCM_USB_response_handler);

		channelThread = new QThread();
		threadList->append(channelThread);

		connect(channelThread, &QThread::finished, channel, &QObject::deleteLater);
		channel->moveToThread(channelThread);

		channelThread->start();
	}
}

RF_System_Minimal::~RF_System_Minimal()
{
	for (int i = 0; i < channel_count; i++)
	{
		threadList->at(i)->quit();
		threadList->at(i)->wait();
	}
}

int RF_System_Minimal::channelCount()
{
	return channel_count;
}

/**********************************************************************************************************************************************************************************
 * Remote Command Mode (blind): Exchange messages between RCM port and SGX port.
 *********************************************************************************************************************************************************************************/
//readyRead handler - if RCM_USB_port is ready to be Read, read the message and send it right along to the SGX port.
void RF_System_Minimal::RCM_USB_message_handler_blind()
{
	emit signal_RCM_USB_message(RCM_USB_port->RCM_Read());
}

void RF_System_Minimal::RCM_USB_response_handler_blind(QString response)
{
// 	#error you are here this connect should be place elsewhere in the right place!
//	connect(RCM_USB_port->serial, &QSerialPort::readyRead, this, &RF_System_1channel_ODC::RCM_USB_response_handler_blind);
	RCM_USB_port->RCM_Write(response);
}

/**********************************************************************************************************************************************************************************
 * Remote Command Mode: Exchange messages between RCM port and SGX port.
 *********************************************************************************************************************************************************************************/
//readyRead handler

//
// TODO:
// Error clearing must be disabled in live RCM!
//
void RF_System_Minimal::RCM_USB_message_handler()
{
	static QString received_message = "";

	received_message += RCM_USB_port->serialRead();
	if ((received_message.contains("\r\n") || received_message.contains("\r") || received_message.contains("\n")) && received_message.contains("$"))
	{
		qDebug() << "RCM USB TX:\t" << received_message;
	}
	else
	{
		if (received_message == "\r\n" || received_message == "\r" || received_message == "\n")
		{
			received_message = ""; //Message was just a blank space, nothing more; clear the variable
		}
		return;
	}

	/* Make sure only one command is handled at a time */
	if (received_message.count("$") > 1)
	{
		RCM_USB_port->serialWrite("RCM: One command at a time please.\r\n");
		received_message = ""; //Message was intercepted; clear the variable
		return;
	}

	if (!received_message.contains(QRegExp("\\$\\w{1,}\r*\n*\r*")))
	{
		qDebug() << "RCM: Command not valid.";
		RCM_USB_port->serialWrite("RCM: Command not valid.\r\n");
		received_message = ""; //Message was intercepted; clear the variable
		return;
	}

	// Disallow commands with multi-line returns and other stuff that doesn't conform to the protocol properly.
	QStringList RCM_blacklist
	{
		"HELP", "HELPDEV",
		"CHANS", "CHANG",
		"IDLES",
		"EECSP", "EECSA"
	};

	for (int i = 0; i < RCM_blacklist.count(); i++)
	{
		if (received_message.contains(("$"+RCM_blacklist.at(i)), Qt::CaseInsensitive))
		{
			qDebug() << "RCM: This command is currently not supported by RCM";
			RCM_USB_port->serialWrite("RCM: This command is currently not supported by RCM\r\n");
			received_message = ""; //Message was intercepted; clear the variable
			return;
		}
	}

	/* Special workaround for S11 sweeping, because it's just too important a feature to not support over RCM */
	if (received_message.contains("$SWP") || received_message.contains("$SWPD"))
	{
		QStringList received_message_list = received_message.split(",");

		/* Intercept sweep request with output mode 1; has multi-line response which is currently not supported over RCM */
		//example of message: $SWP,0,2400,2500,10,50,0\r\n
		if (received_message_list.count() == 7)
		{
			if (received_message_list.last().toInt() == 0)
			{
				emit signal_RCM_sweep_message(received_message);
				received_message = ""; //Message was intercepted; clear the variable
				return;
			}
		}
	}

	emit signal_RCM_USB_message(received_message);
	received_message = ""; //Message was sent; clear the variable
}

void RF_System_Minimal::RCM_USB_response_handler(QString response)
{
	//Workaround: Remove /r/n from the response, so that the /r/n that serialWrite is about to add is not redundant.
	response.chop(2);	
	
	RCM_USB_port->serialWrite(response);
}

void RF_System_Minimal::RCM_USB_error_handler(QSerialPort::SerialPortError error)
{
//	if (error != QSerialPort::SerialPortError::NoError)
//	{
//		if (RCM_USB_port->serial->isOpen())
//		{
//			RCM_USB_port->close();
//		}

//		RCM_USB_port->open();
//	}
}
